// pages/api/admin/roles/index.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { z } from "zod";

const CreateRoleSchema = z.object({
  name: z.string().trim().min(1).max(64),
  description: z.string().trim().max(255).optional(),
});

const UpdateRoleSchema = z.object({
  name: z.string().trim().min(1).max(64).optional(),
  description: z.string().trim().max(255).optional(),
  isActive: z.boolean().optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method === "GET") {
    // Parse query parameters for filters
    const { search, isActive } = req.query;

    let where: any = {};

    // Search filter
    if (search) {
      where.OR = [
        { name: { contains: search as string, mode: "insensitive" } },
        { description: { contains: search as string, mode: "insensitive" } },
      ];
    }

    // Active filter
    if (isActive !== undefined) {
      where.isActive = isActive === "true";
    }

    const roles = await prisma.role.findMany({
      where,
      orderBy: { name: "asc" },
      include: {
        _count: {
          select: { users: true },
        },
      },
    });

    return res.status(200).json({ ok: true, roles });
  }

  if (req.method === "POST") {
    const parsed = CreateRoleSchema.safeParse(req.body);
    if (!parsed.success) {
      return res.status(400).json({
        ok: false,
        error: "Invalid data",
        details: parsed.error.errors,
      });
    }

    try {
      const role = await prisma.role.create({
        data: {
          name: parsed.data.name,
          description: parsed.data.description,
        },
        include: {
          _count: {
            select: { users: true },
          },
        },
      });

      return res.status(201).json({ ok: true, role });
    } catch (error: any) {
      if (error.code === "P2002") {
        return res.status(400).json({
          ok: false,
          error: "A role with this name already exists",
        });
      }
      return res.status(500).json({
        ok: false,
        error: "Failed to create role",
      });
    }
  }

  return res.status(405).json({ ok: false, error: "Method not allowed" });
}
